/* --------------------------------------------------------------
  StyleUpload.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Modal from '../libs/Modal';

/**
 * Style Upload Widget
 *
 * This widget will use the jquery-fileupload plugin in order to upload the style
 * to the server with POST request into a hidden iframe.
 */
export default class StyleUpload {
	/**
	 * Class Constructor
	 *
	 * @param {jQuery} $target The target selector to be converted into an StyleUpload element. This
	 * must be a input:file element.
	 * @param {String} url The URL that will be used to POST the file.
	 */
	constructor($target, url) {
		StyleEdit.Validator.isObject($target);
		StyleEdit.Validator.isString(url);
		
		/**
		 * The to be converted into a jquery ajax-handler.
		 *
		 * @type {jQuery}
		 */
		this.$target = $target;
		
		/**
		 * URL to be used on upload.
		 *
		 * @type {String}
		 */
		this.url = url;
	}
	
	/**
	 * Trigger the file upload.
	 *
	 * Since this class is a wrapper for the jquery-fileupload plugin it will handle the AJAX
	 * failure on its own.
	 *
	 * @param {String} styleName The name of the new style.
	 *
	 * @return {jQuery.Promise} Returns a promise object that will be resolved with the server's response.
	 */
	upload(styleName) {
		const deferred = $.Deferred();
		const filesList = this.$target.get(0).files;
		
		this.$target.fileupload({
			formData: {style_name: styleName},
			url: this.url,
			dataType: 'json',
			autoUpload: false,
			acceptFileTypes: /(\.|\/)(json)$/i,
			forceIframeTransport: true
		});
		
		this.$target.fileupload('send', {files: filesList})
			.done(result => {
				deferred.resolve(result);
			})
			.fail((jqxhr, testStatus, errorThrown) => {
				Modal.message(StyleEdit.Language.translate('title_upload_failure', 'style_edit'),
					StyleEdit.Language.translate('message_upload_failure', 'style_edit'));
				StyleEdit.Debug.error('StyleUpload Widget Error: ', errorThrown, testStatus, jqxhr);
			});
		
		return deferred.promise();
	}
}
